COMMENT
Na/K pump from Takeuchi et al., 2006. 
Model code was modified from Botta et al., 2010. 
ENDCOMMENT

NEURON{
    SUFFIX NaKpump
    USEION na READ nai, nao WRITE ina
    USEION k READ ki, ko WRITE ik
    RANGE imax, flux, yy
}

UNITS{
    (mV) = (millivolt)
    (mA) = (milliamp)
    (molar) = (1/liter)
    (mM) = (millimolar)
    FARADAY = (faraday) (coulomb)
}

PARAMETER{
    temperature = 27 (degC)
    imax = 0.006 (mA/cm2)
    RR = 8.314463 (joule/K/molar)
    KdNao = 69.8
    KdKo = 0.258
    KdNai = 4.05
    KdKi = 32.88
    ATPi = 5 (mM)
    k2 = 0.04
    k3 = 0.01
    k4 = 0.165
}

ASSIGNED{
    v (mV)
    ina (mA/cm2)
    ik (mA/cm2)
    flux (mA/cm2)
    nai (mM)
    nao (mM)
    ki (mM)
    ko (mM)
    Naeff (mM)
    k1
    pE1Na
    pE1K
    pE2Na
    pE2K
    alphay (1/ms)
    betay (1/ms)
    kelvin (K)
}

STATE{
    yy
}

INITIAL{
    settables(nai, nao, ki, ko, v, temperature, ATPi)
	yy = alphay/(alphay+betay) :initialize at steady state
}

BREAKPOINT{
    SOLVE states METHOD cnexp
    settables (nai, nao, ki, ko, v, temperature, ATPi)
    flux = imax * (k1*pE1Na*yy - k2*pE2Na*(1-yy))
    ina = 3 * flux
    ik = -2 * flux
}

DERIVATIVE states {
	settables(nai, nao, ki, ko, v, temperature, ATPi)
	yy' = alphay*(1-yy) - betay*yy
}

PROCEDURE settables(nai (mM), nao (mM), ki (mM), ko (mM), v (mV), temperature (degC), ATPi (mM)) {
    UNITSOFF
    k1 = 0.37 * (1/(1+0.094/ATPi))  
	kelvin = temperature + 273.16  
	Naeff = nao * exp(-0.82*FARADAY*v/RR/kelvin/1000)
	pE1Na = 1 / (1 + (KdNai/nai)^1.06*(1+(ki/KdKi)^1.12))
	pE1K = 1 / (1 + (KdKi/ki)^1.12*(1+(nai/KdNai)^1.06))
	pE2Na = 1 / (1 + (KdNao/Naeff)^1.06*(1+(ko/KdKo)^1.12))
	pE2K = 1 / (1 + (KdKo/ko)^1.12*(1+(Naeff/KdNao)^1.06))
    alphay = k2*pE2Na + k4*pE2K
	betay = k1*pE1Na + k3*pE1K
    UNITSON
}
